#include"MC9S08GW64.h"
#include"iic.h"

void (*callback)(void);    //This is a pointer to the function which 
                           //the user needs to pass in IIC_Init
                           //so that the required function is 
                           //called in interrupt subroutine


/******************************************************************************/
/* FUNCTION          : IIC_Init                                               */
/******************************************************************************/
/* Abstract          : This function initializes i2c interface                */
/*					           by configuring the internal registers                  */
/*                                                                            */
/* Input Parameters  : Interrupt_Enable: IIC_INTERRUPT_ENABLE                 */
/*                                       /IIC_INTERRUPT_DISABLE               */
/*                     Addr_No_Of_Bits : IIC_10_BIT_SLAVE_ADDR                */ 
/*                                       /IIC_7_BIT_SLAVE_ADDR                */
/*                                                                            */
/* Return Parameter  : None                                                   */
/******************************************************************************/
void IIC_Init(unsigned char Interrupt_Enable,unsigned char Addr_No_Of_Bits,void (*p)(void)) 
{
  SCGC1_IIC =1 ;   //enables the clock for IIC module
  callback = p;
  //pin muxing
  if(IIC_SDA_SCA_PortA) 
    { 
        PTAPF1_A0 = 0b011;  //PTA0 is SCL 
        PTAPF1_A1 = 0b011;  //PTA1 is SDA
    }
   else
    {
        PTBPF3_B4 = 0b011;  //PTB4 is SCL 
        PTBPF3_B5 = 0b011;  //PTB5 is SDA
    }
 
  IICC1_IICIE = Interrupt_Enable;   //if Interrupt_EN = 1 then IIC interrupt is enabled
 
  IICC2_ADEXT = Addr_No_Of_Bits;     //if 10Bit_Addr = 1 then slave address is of 10 bits
                               //if 10Bit_Addr = 0 then slave address is of 7 bits
                               
  //Baud Rate = (Bus Clk/(MULT * SCL DIVIDER))
  /* set MULT and ICR for baud rate 250Khz if Bus Clk is 20 MHz*/ 

  IICF  = 0x14;       // set MULT and ICR for baud rate 250Khz                                     
 
  IICC1_IICEN = 1;   //enables the IIC module
}
 


/******************************************************************************/
/* FUNCTION          : IIC_Wait_Ack                                      */
/******************************************************************************/
/* Abstract          : This function is called when waiting for some data/ack */
/*					           It waits on IIC flag and clears it afterwards          */
/*                                                                            */
/* Input Parameters  : None                                                   */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/
void IIC_Wait_Ack(void)
{
 while(!IICS_IICIF); 
  //poll for ACK Receive
 
   while(IICS_RXAK);
  
   //clear IICIF BIT
   IICS_IICIF =1;
}

/******************************************************************************/
/* FUNCTION          : IIC_StartTransmission                                  */
/******************************************************************************/
/* Abstract          : This function starts the transaction of read/write     */
/*					           with a slave                                           */
/*                                                                            */
/* Input Parameters  : SlaveAddress: Address of destination slave             */
/*                     TxRxMode: Specifies the mode Tx or Rx                  */
/*                                                                            */
/* Return Parameter  : None                                                   */
/*                                                                            */
/* Traceability Info :                                                        */
/******************************************************************************/
void IIC_Start_Transmission(unsigned int SlaveAddress,unsigned char TxRxMode)
{    
  unsigned char Slave_ID;
  unsigned char temp1;
  unsigned char temp2;
  
  if(IICC2_ADEXT)   //if 10 bit address mode 
  {
    temp1 = 0b11110110;
    temp1 |= TxRxMode;
    temp2 = (unsigned char) (SlaveAddress >> 7);
    temp2 |= 0b11111001;
    temp2 &= temp1;
  }
  
  /* shift ID in right possition */
  Slave_ID = (unsigned char)SlaveAddress << 1;    //LSB 7 bits of slave address
  
  /* Set R/W bit at end of Slave Address */
  Slave_ID |= TxRxMode;                      //Type of transaction -> Tx or Rx

  /* send start signal */
  IIC_Send_Start();
  
  if(IICC2_ADEXT) 
  {
    IIC_Write_Byte(temp2);
    IIC_Write_Byte((unsigned char)(SlaveAddress));
    return;
  }

  /* send ID with W/R bit */
  IIC_Write_Byte(Slave_ID);
}

/*Interrupt Sub Routine*/
void interrupt VectorNumber_Viic IIC_ISR() 
{
 IICS_IICIF = 1;                     /* clear IIC interrupt flag */
 if((*callback)) 
 {
   callback();
 }
 //code to be written
 
}